#include <Pythia8/Pythia.h>
#include <iostream>
#include <ctime>

using namespace Pythia8;

int main() {

    Pythia pythia;
    pythia.readFile("main001.cmnd");
    int nEvent = pythia.mode("Main:numberOfEvents");
    pythia.init();


    Hist dileptonMass("Dilepton Invariant Mass (m_ll)", 100, 50.0, 150.0);
    Hist dileptonRapidity("Dilepton Rapidity (y_ll)", 100, -5.0, 5.0);
    Hist dileptonPt("Dilepton Transverse Momentum (pT_ll)", 100, 0.0, 200.0);


    int nFilled = 0;

    auto t1 = std::time(0);
    // Event loop
    for (int iEvent = 0; iEvent < nEvent; ++iEvent) {
        if (!pythia.next()) continue;

        std::vector<Particle> muons;
        for (int i = 0; i < pythia.event.size(); ++i) {
            if (pythia.event[i].isFinal() && abs(pythia.event[i].id()) == 13) {
                muons.push_back(pythia.event[i]);
            }
        }

        if (muons.size() == 2 && muons[0].id() * muons[1].id() < 0) {
            Vec4 p1 = muons[0].p();
            Vec4 p2 = muons[1].p();
            Vec4 pDilepton = p1 + p2;

            double mass = pDilepton.mCalc();
            double rapidity = pDilepton.rap();
            double pt = pDilepton.pT();

            dileptonMass.fill(mass);
            dileptonRapidity.fill(rapidity);
            dileptonPt.fill(pt);
            nFilled++;
        }
    }
    auto t2 = std::time(0);
    std::cout << "Total time: " << t2 - t1 << " seconds" << std::endl;


    pythia.stat();


    double sigma = pythia.info.sigmaGen(); // in mb
    double binWidthMass = (150.0 - 50.0) / 100;     // 1.0 GeV
    double binWidthRapidity = (5.0 - (-5.0)) / 100; // 0.1
    double binWidthPt = (200.0 - 0.0) / 100;        // 2.0 GeV

    // Scale histograms to differential cross sections
    if (nFilled > 0) {
        dileptonMass *= sigma / nFilled / binWidthMass;
        dileptonRapidity *= sigma / nFilled / binWidthRapidity;
        dileptonPt *= sigma / nFilled / binWidthPt;
    } else {
        std::cout << "Warning: No events filled the histograms." << std::endl;
    }


    std::cout << "sigma = " << sigma << " mb, nFilled = " << nFilled << std::endl;
    std::cout << dileptonMass;
    std::cout << dileptonRapidity;
    std::cout << dileptonPt;
    HistPlot hpl("plotDrell");
    hpl.frame( "plotDrell", "mass", "$n_{\\mathrm{charged}}$",
      "Rate");
    hpl.add( dileptonMass, "-,blue", "default");
    hpl.plot();
    hpl.frame( "", "dilepton rapidity", "$y$", "Rate");
    hpl.add( dileptonRapidity, "-,blue", "default");
    hpl.plot();
    hpl.frame( "", "dilepton rapidity", "$p_{\\perp}$", "Rate");
    hpl.add( dileptonPt, "-,blue", "default");
    hpl.plot();
    return 0;
}
