// PDFxTMD.h is a part of the PYTHIA event generator.
// Copyright (C) 2025 Torbjorn Sjostrand.
// PYTHIA is licenced under the GNU GPL v2 or later, see COPYING for details.
// Please respect the MCnet Guidelines, see GUIDELINES for details.

// This file contains the PDFxTMD PDF plugin class.

#ifndef Pythia8_PDFxTMD_PDF_H
#define Pythia8_PDFxTMD_PDF_H

#include "Pythia8/PartonDistributions.h"
#include "Pythia8/Plugins.h"
#include <PDFxTMDLib/Factory.h>
#include <PDFxTMDLib/PDFSet.h>
#include <PDFxTMDLib/Common/ConfigWrapper.h>
#include <mutex>
#include <memory>
#include <iostream>

namespace Pythia8 {

//==========================================================================

// Containers for PDF sets.

//--------------------------------------------------------------------------

//==========================================================================

// Provide interface to the LHAPDF6 library of parton densities.



class PDFxTMD_PDF : public PDF {

public:
  static PDFxTMD::PartonFlavor MinusPDFsToAntiPartonFlavor(int idn)
  {
    switch(idn)
    {
      case -1:
        return PDFxTMD::dbar;
      case -2:
        return PDFxTMD::ubar;
      case -3:
        return PDFxTMD::sbar;
      case -4:
        return PDFxTMD::cbar;
      case -5:
        return PDFxTMD::bbar;
      case -6:
        return PDFxTMD::tbar;
      default:
        throw std::runtime_error("Unkown pdf number. Cannot convert to PDFxTMD::PartonFlavor");
    }
  }
  // Constructor.
  PDFxTMD_PDF(Pythia*, Settings* settingsPtr, Logger*) :
    PDF(), m_extrapol(false), m_pdf(nullptr){
    if (settingsPtr == nullptr) return;
    sSymmetric(settingsPtr->flag("LHAPDF:sSymmetric"));
    cSymmetric(settingsPtr->flag("LHAPDF:cSymmetric"));
    bSymmetric(settingsPtr->flag("LHAPDF:bSymmetric"));
  }

  // Initialization of PDF set.
  bool init(int idBeamIn, string setName, int member, Logger* loggerPtr)
    override;

  // Allow extrapolation beyond boundaries (not implemented).
  void setExtrapolate(bool extrapolIn) {m_extrapol = extrapolIn;}

private:

  // The LHAPDF objects.
  std::unique_ptr<::PDFxTMD::PDFSet<PDFxTMD::CollinearPDFTag>> m_pdfs;
  ::PDFxTMD::ICPDF* m_pdf;
  bool m_extrapol;

  // Update parton densities.
  void xfUpdate(int id, double x, double Q2);

  // Check whether x and Q2 values fall inside the fit bounds.
  bool insideBounds(double x, double Q2) {
    return (x > xMin && x < xMax && Q2 > q2Min && Q2 < q2Max);}

  // Return the running alpha_s shipped with the LHAPDF set.
  double alphaS(double Q2) { return m_pdfs->alphasQ2(Q2); }

  // Return quark masses used in the PDF fit.
  double muPDFSave, mdPDFSave, mcPDFSave, msPDFSave, mbPDFSave,
         xMin, xMax, q2Min, q2Max;
  double mQuarkPDF(int id) {
    switch(abs(id)){
      case 1: return mdPDFSave;
      case 2: return muPDFSave;
      case 3: return msPDFSave;
      case 4: return mcPDFSave;
      case 5: return mbPDFSave;
    }
    return -1.;
 }

  // Calculate uncertainties using the LHAPDF prescription.
  void calcPDFEnvelope(int, double, double, int);
  void calcPDFEnvelope(pair<int,int>, pair<double,double>, double, int);
  PDFEnvelope pdfEnvelope;
  PDFEnvelope getPDFEnvelope() {return pdfEnvelope;}
  static const double PDFMINVALUE;

  int nMembersSave;
  int nMembers() { return nMembersSave; }

};

//--------------------------------------------------------------------------

// Constants.

const double PDFxTMD_PDF::PDFMINVALUE = 1e-10;

//--------------------------------------------------------------------------

// Initialize a parton density function from LHAPDF6.

bool PDFxTMD_PDF::init(int idBeamIn, string setName, int member,
  Logger* loggerPtr) {
  idBeam = idBeamIn;
  idBeamAbs = abs(idBeamIn);
  isSet = false;

  // Find the PDF set. Note, LHAPDF aborts if the PDF does not exist,
  // which we avoid with this try/catch statement. Ideally, we would
  // check with :LHAPDF::lookupLHAPDFID, but this is not thread safe.
  try {
     m_pdfs =  std::make_unique<PDFxTMD::PDFSet<PDFxTMD::CollinearPDFTag>>((setName));
  } catch (const std::exception &e) {
    loggerPtr->ERROR_MSG("unknown PDF " + setName);
    return false;
  }

  // Find the PDF member.
  if (m_pdfs->size() == 0) {
    loggerPtr->ERROR_MSG("could not initialize PDF " + setName);
    return false;
  } else if (member >= m_pdfs->size()) {
    loggerPtr->ERROR_MSG(setName + " does not contain requested member");
    return false;
  }
  m_pdf = (*m_pdfs)[member];
  isSet = true;

  // Save x and Q2 limits.
  xMax  = m_pdfs->getStdPDFInfo().XMax;
  xMin  = m_pdfs->getStdPDFInfo().XMin;
  q2Max = SQR(m_pdfs->getStdPDFInfo().QMax);
  q2Min = SQR(m_pdfs->getStdPDFInfo().QMin);

  // Store quark masses used in PDF fit.
  auto MUpValuePair = m_pdfs->info().get<double>("MUp");
  double mupValue = 0;
  if (MUpValuePair.first != std::nullopt)
  {
    mupValue = MUpValuePair.first.value();
  }
  auto MDownValuePair = m_pdfs->info().get<double>("MDown");
  double MDownValue = 0;
  if (MDownValuePair.first != std::nullopt)
  {
    MDownValue = MDownValuePair.first.value();
  }
  auto MStrangeValuePair = m_pdfs->info().get<double>("MStrange");
  double MStrangeValue = 0;
  if (MStrangeValuePair.first != std::nullopt)
  {
    MStrangeValue = MStrangeValuePair.first.value();
  }
  auto MCharmValuePair = m_pdfs->info().get<double>("MCharm");
  double MCharmValue = 0;
  if (MCharmValuePair.first != std::nullopt)
  {
    MCharmValue = MCharmValuePair.first.value();
  }
  auto MBottomValuePair = m_pdfs->info().get<double>("MBottom");
  double MBottomValue = 0;
  if (MBottomValuePair.first != std::nullopt)
  {
    MBottomValue = MBottomValuePair.first.value();
  }
  muPDFSave = mupValue;
  mdPDFSave = MDownValue;
  mcPDFSave = MCharmValue;
  msPDFSave = MStrangeValue;
  mbPDFSave = MBottomValue;
  nMembersSave  = m_pdfs->getStdPDFInfo().NumMembers;
  return true;

}

//--------------------------------------------------------------------------

// Give the parton distribution function set from LHAPDF6.

void PDFxTMD_PDF::xfUpdate(int, double x, double Q2) {
  if (!isSet) return;

  // Freeze at boundary value if PDF is evaluated outside the fit region.
  if (x < xMin && !m_extrapol) x = xMin;
  if (x > xMax)    x = xMax;
  if (Q2 < q2Min) Q2 = q2Min;
  if (Q2 > q2Max) Q2 = q2Max;

  // Update values.
  xg     = m_pdf->pdf(PDFxTMD::PartonFlavor::g, x, Q2);
  xd     = m_pdf->pdf(PDFxTMD::PartonFlavor::d,  x, Q2);
  xu     = m_pdf->pdf(PDFxTMD::PartonFlavor::u,  x, Q2);
  xdbar  = m_pdf->pdf(PDFxTMD::PartonFlavor::dbar, x, Q2);
  xubar  = m_pdf->pdf(PDFxTMD::PartonFlavor::ubar, x, Q2);
  xs     = m_pdf->pdf(PDFxTMD::PartonFlavor::s,  x, Q2);
  xc     = m_pdf->pdf(PDFxTMD::PartonFlavor::c,  x, Q2);
  xb     = m_pdf->pdf(PDFxTMD::PartonFlavor::b,  x, Q2);
  xsbar  = sSymmetricSave ? xs : m_pdf->pdf(PDFxTMD::PartonFlavor::sbar, x, Q2);
  xcbar  = cSymmetricSave ? xc : m_pdf->pdf(PDFxTMD::PartonFlavor::cbar, x, Q2);
  xbbar  = bSymmetricSave ? xb : m_pdf->pdf(PDFxTMD::PartonFlavor::bbar, x, Q2);
  xgamma = m_pdf->pdf(PDFxTMD::PartonFlavor::photon, x, Q2);
  // idSav = 9 to indicate that all flavours reset.
  idSav = 9;

}

//--------------------------------------------------------------------------

// Calculate uncertainties using the LHAPDF prescription.

void PDFxTMD_PDF::calcPDFEnvelope(int idNow, double xNow, double Q2NowIn,
  int valSea) {
  if (!isSet) return;

  // Freeze at boundary value if PDF is evaluated outside the fit region.
  double x1 = (xNow < xMin && !m_extrapol) ? xMin : xNow;
  if (x1 > xMax) x1 = xMax;
  double Q2Now = (Q2NowIn < q2Min) ? q2Min : Q2NowIn;
  if (Q2Now > q2Max) Q2Now = q2Max;

  // Loop over the members.
  vector<double> xfCalc(m_pdfs->size());
  for(int iMem = 0; iMem < m_pdfs->size(); ++iMem) {
    if (valSea==0 || (idNow != 1 && idNow != 2)) {
      xfCalc[iMem] = (*m_pdfs)[iMem]->pdf(static_cast<PDFxTMD::PartonFlavor>(idNow), x1, Q2Now);
    } else if (valSea==1 && (idNow == 1 || idNow == 2 )) {
      xfCalc[iMem] = (*m_pdfs)[iMem]->pdf(static_cast<PDFxTMD::PartonFlavor>(idNow), x1, Q2Now) -
        (*m_pdfs)[iMem]->pdf(MinusPDFsToAntiPartonFlavor(-idNow), x1, Q2Now);
    } else if (valSea==2 && (idNow == 1 || idNow == 2 )) {
      xfCalc[iMem] = (*m_pdfs)[iMem]->pdf(MinusPDFsToAntiPartonFlavor(-idNow), x1, Q2Now);
    }
  }

  // Calculate the uncertainty.
  ::PDFxTMD::PDFUncertainty xfErr = m_pdfs->Uncertainty(xfCalc);
  pdfEnvelope.centralPDF = xfErr.central;
  pdfEnvelope.errplusPDF = xfErr.errplus;
  pdfEnvelope.errminusPDF = xfErr.errminus;
  pdfEnvelope.errsymmPDF = xfErr.errsymm;
  pdfEnvelope.scalePDF = xfErr.scale;
}

//--------------------------------------------------------------------------

// Calculate uncertainties using the LHAPDF prescription.

void PDFxTMD_PDF::calcPDFEnvelope(pair<int,int> idNows, pair<double,double> xNows,
  double Q2NowIn, int valSea) {
  if (!isSet) return;

  // Freeze at boundary value if PDF is evaluated outside the fit region.
  double x1 = (xNows.first < xMin && !m_extrapol) ? xMin : xNows.first;
  if (x1 > xMax) x1 = xMax;
  double x2 = (xNows.second < xMin && !m_extrapol) ? xMin : xNows.second;
  if (x2 > xMax) x2 = xMax;
  double Q2Now = (Q2NowIn < q2Min) ? q2Min : Q2NowIn;
  if (Q2Now > q2Max) Q2Now = q2Max;

  // Loop over the members.
  vector<double> xfCalc(m_pdfs->size());
  pdfEnvelope.pdfMemberVars.resize(m_pdfs->size());
  for(int iMem = 0; iMem < m_pdfs->size(); ++iMem) {
    if        (valSea == 0 || (idNows.first != 1 && idNows.first != 2 ) ) {
      xfCalc[iMem] = (*m_pdfs)[iMem]->pdf(static_cast<PDFxTMD::PartonFlavor>(idNows.first), x1, Q2Now);
    } else if (valSea == 1 && (idNows.first == 1 || idNows.first == 2)) {
      xfCalc[iMem] = (*m_pdfs)[iMem]->pdf(static_cast<PDFxTMD::PartonFlavor>(idNows.first), x1, Q2Now)
        - (*m_pdfs)[iMem]->pdf(MinusPDFsToAntiPartonFlavor(-idNows.first), x1, Q2Now);
    } else if (valSea == 2 && (idNows.first == 1 || idNows.first == 2 )) {
      xfCalc[iMem] = (*m_pdfs)[iMem]->pdf(MinusPDFsToAntiPartonFlavor(-idNows.first), x1, Q2Now);
    }
    xfCalc[iMem] = max(0.0, xfCalc[iMem]);
    if        (valSea == 0 || (idNows.second != 1 && idNows.second != 2)) {
      xfCalc[iMem] /= max
        (PDFMINVALUE, (*m_pdfs)[iMem]->pdf(static_cast<PDFxTMD::PartonFlavor>(idNows.second), x2, Q2Now));
    } else if (valSea == 1 && (idNows.second == 1 || idNows.second == 2 )) {
      xfCalc[iMem] /= max
        ((*m_pdfs)[iMem]->pdf(static_cast<PDFxTMD::PartonFlavor>(idNows.second), x2, Q2Now) - (*m_pdfs)[iMem]->pdf
         (MinusPDFsToAntiPartonFlavor(-idNows.second), x2, Q2Now), PDFMINVALUE);
    } else if (valSea == 2 && (idNows.second == 1 || idNows.second == 2 )) {
      xfCalc[iMem] /= max
        ((*m_pdfs)[iMem]->pdf(MinusPDFsToAntiPartonFlavor(-idNows.second), x2, Q2Now), PDFMINVALUE);
    }
    pdfEnvelope.pdfMemberVars[iMem] = xfCalc[iMem];
  }

  // Calculate the uncertainty.
  ::PDFxTMD::PDFUncertainty xfErr = m_pdfs->Uncertainty(xfCalc);
  pdfEnvelope.centralPDF = xfErr.central;
  pdfEnvelope.errplusPDF = xfErr.errplus;
  pdfEnvelope.errminusPDF = xfErr.errminus;
  pdfEnvelope.errsymmPDF = xfErr.errsymm;
  pdfEnvelope.scalePDF = xfErr.scale;

}

//--------------------------------------------------------------------------

// Declare the plugin.

PYTHIA8_PLUGIN_CLASS(PDF, PDFxTMD_PDF, false, false, false)
PYTHIA8_PLUGIN_VERSIONS(PYTHIA_VERSION_INTEGER)

//==========================================================================

} // end namespace Pythia8

#endif // end Pythia8_PDFxTMD_PDF_H
